<?php
/* --------------------------------------------------------------
   HubPayPalSettings.inc.php 2023-02-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2022 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

use Gambio\Core\Cache\Services\CacheFactory;
use HubPublic\Exceptions\CurlRequestException;
use HubPublic\Http\CurlRequest;
use League\Container\Exception\NotFoundException;

class HubPayPalSettings
{
    protected $config;
    protected $cache;
    
    const CACHE_TTL = 3600;
    const CACHE_FILE = 'paypal2hub.json.cache';
    
    public function __construct(HubPayPalConfiguration $configuration)
    {
        $this->config = $configuration;
        $this->cache = null;
        if (class_exists(LegacyDependencyContainer::class)) {
            try {
                /** @var CacheFactory $cacheFactory */
                $cacheFactory = LegacyDependencyContainer::getInstance()->get(CacheFactory::class);
                $this->cache = $cacheFactory->createCacheFor('paypal2hub');
            } catch (NotFoundException $e) {
                $this->cache = null;
            }
        }
    }
    
    
    protected function getJavascriptSource($position = 'cart') : array
    {
        $result = [
            'useVault' => false,
            'usePayNow' => false,
            'jssrc' => '',
            'partnerAttributionId' => ''
        ];
        if (empty(trim(gm_get_conf('GAMBIO_HUB_CLIENT_KEY')))) {
            return $result; // shop is not connected to the hub
        }
        
        if (!gm_get_conf('GAMBIO_HUB_REMOTE_CONFIG_PAYPAL2HUB_ECS')) {
            return $result; // The module is not installed.
        }
        $cacheKey = 'paypalhub_jssrc_' . $position;
        try {
            $refreshCache = false;
            $jsSrcPayload = $this->getJavascriptSourceFromCache($cacheKey);
            if ($jsSrcPayload === null) {
                $jsSrcPayload = $this->getJavascriptSourceFromHub($position);
                $refreshCache = true;
            }
            $url = parse_url($jsSrcPayload['jssrc']);
            parse_str($url['query'], $query);
            $query['currency'] = $_SESSION['currency'];
            $payNow = filter_var($jsSrcPayload['usePayNow'] ?? false, FILTER_VALIDATE_BOOLEAN);
            if ((basename($_SERVER['SCRIPT_FILENAME']) === 'checkout_payment.php') && $payNow === true) {
                $query['commit'] = 'true';
            }
            $result['usePayNow'] = $payNow;
            if (strpos($_SERVER['SCRIPT_NAME'], 'shopping_cart') === false) {
                unset($query['disable-funding']);
            }
            $result['partnerAttributionId'] = $jsSrcPayload['partnerAttributionId'] ?? null;
            $builtQuery = http_build_query($query, '', '&');
            $useVault = $jsSrcPayload['useVault'] ?? false;
            if ($useVault == 'true') {
                $result['useVault'] = true;
            }
            $result['jssrc'] ="{$url['scheme']}://{$url['host']}{$url['path']}?{$builtQuery}";
            if ($refreshCache === true) {
                $this->setJavascriptSourceInCache($cacheKey, $jsSrcPayload);
            }
            return $result; 
        } catch (Exception $e) {
            return $result;
        }
    }

    /**
     * @return string|null
     */
    public function getUserVaultTokenFromSession() : ? string
    {
        // get vault setting from session
        $customerVault = $_SESSION['customer_vault'] ?? null;
        
        // not cache yet, get from Hub
        if (!$customerVault) {
            $customerVault = $this->getUserVaultTokenFromHub();
            $_SESSION['customer_vault'] = $customerVault;
        }
        // warm up vault token when user selected vault account. 
        if (
            !isset($customerVault['tokenPayload']['expired_at']) ||
            (!$customerVault['customerUseVault'] && (basename($_SERVER['SCRIPT_FILENAME']) === 'checkout_success.php'))
        ) {
            unset($_SESSION['customer_vault']);
            return null;
        }
        
        // no using vault use on this session
        if (!$customerVault['customerUseVault']) {
            return null;
        }
        
        // check current token still valid
        $now = time();
        $expiresAt = $customerVault['tokenPayload']['expired_at'];
        
        // token expired so refresh
        if ($now > $expiresAt) {
            unset($_SESSION['customer_vault']);
            return null;
        }
        
        return $customerVault['tokenPayload']['token'];
    }

    /**
     *  Get user vault token from hub
     * 
     * @return array
     */
    private function getUserVaultTokenFromHub(): array
    {
        $result = [
            'customerUseVault' => false,
            'tokenPayload' => null,
        ];
        $query = [
            'client_key' => gm_get_conf('GAMBIO_HUB_CLIENT_KEY'),
            'customer_id' => $_SESSION['customer_id']
        ];

        $maxTimeout = 5;
        $timeout = min($maxTimeout, (int)gm_get_conf('GAMBIO_HUB_CURL_TIMEOUT'));
        $hubSettings = MainFactory::create('HubSettings', $timeout);
        $hubCallbackApiClient = MainFactory::create(
            'HubCallbackApiClient',
            MODULE_PAYMENT_GAMBIO_HUB_URL,
            new CurlRequest(),
            LogControl::get_instance(),
            $hubSettings
        );
        $params = ['source' => 'get_user_vault_token'];
        $response = $hubCallbackApiClient->execute('PayPal2Hub', true, $params, $query);
        if ($response->getStatusCode() === 200) {
            $responseBody = json_decode($response->getBody(), true);
            if (!$responseBody['vaulted']) {
                return $result;
            }
            if (is_array($responseBody) && !empty($responseBody['tokenKey'])) {
                $vaultData = $responseBody['tokenPayload'] ?? null;
                if ($vaultData) {
                    $result['customerUseVault'] = true;
                    $result['tokenPayload'] = $vaultData;
                }
            }
        }

        return $result;
    }

    /**
     * Get JS sdk url from hub
     * @param $position
     * @return array|null
     */
    protected function getJavascriptSourceFromHub($position = 'cart'): ?array
    {
        $query = [
            'client_key' => gm_get_conf('GAMBIO_HUB_CLIENT_KEY'),
            'devmode'    => file_exists(DIR_FS_CATALOG . '.dev-environment') ? 'true' : 'false',
        ];
        
        $maxTimeout = 5;
        $timeout = min($maxTimeout, (int)gm_get_conf('GAMBIO_HUB_CURL_TIMEOUT'));
        /** @var HubSettings $hubSettings */
        $hubSettings = MainFactory::create('HubSettings', $timeout);
        
        /** @var \HubCallbackApiClient $hubCallbackApiClient */
        $hubCallbackApiClient = MainFactory::create(
            'HubCallbackApiClient',
            MODULE_PAYMENT_GAMBIO_HUB_URL,
            new CurlRequest(),
            LogControl::get_instance(),
            $hubSettings
        );
        
        $params = ['source' => 'get_javascript_source', 'position' => (string)$position];
        /** @var \HubPublic\ValueObjects\HttpResponse $response */
        $response = $hubCallbackApiClient->execute('PayPal2Hub', true, $params, $query);

        if ($response->getStatusCode() !== 200) {
            return null;
        }

        $responseBody = json_decode($response->getBody(), true);
        if (!empty($responseBody['jssrc']) && is_array($responseBody) && json_last_error() === JSON_ERROR_NONE) {
            return $responseBody;
        }

        return null;
    }

    /**
     * Get js SDK url from cache
     * 
     * @param $cacheKey
     * @return mixed|null
     */
    protected function getJavascriptSourceFromCache($cacheKey): ?array
    {
        try {
            // Try to get from cache object if available
            if ($this->cache !== null && $this->cache->has($cacheKey)) {
                $cachedData = $this->cache->get($cacheKey);
            } else {
                // Fallback to file-based cache
                $cachedData = $this->getFromCacheByFileName(static::CACHE_FILE);
            }
            // Not exits cache
            if ($cachedData === null) {
                return null;
            }
    
            // Cache is invalid payload (Could be the old cache)
            if (!isset($cachedData['payload']['useVault'])) {
                $this->removeInCacheByKey($cacheKey);
                return null;
            }
            // Check expiration
            if (!isset($cachedData['expires_at']) || (int)$cachedData['expires_at'] <= time()) {
                return null;
            }
    
            return $cachedData['payload'];
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * Set new cache with vault setting
     * 
     * @param $cacheKey
     * @param $jsSrcPayload
     * @return void
     */
    protected function setJavascriptSourceInCache(string $cacheKey, array $jsSrcPayload)
    {
        $cacheData = [
            'expires_at' => time() + static::CACHE_TTL,
            'payload' => $jsSrcPayload
        ];
        if ($this->cache !== null) {
            $this->cache->set($cacheKey, $cacheData, static::CACHE_TTL);
            return;
        }
        $cacheFile = DIR_FS_CATALOG . 'cache/' . static::CACHE_FILE;
        file_put_contents($cacheFile, json_encode($cacheData));
    }

    /**
     * Get cache from file
     * 
     * @param $fileName
     * @return mixed|null
     */
    private function getFromCacheByFileName($fileName) : ?array
    {
        $cacheFile = DIR_FS_CATALOG . 'cache/' . $fileName;
        if ($this->cache === null && file_exists($cacheFile)) {
            return json_decode(file_get_contents($cacheFile), true);
        }
        return null;
    }

    /**
     * Warm up cache function 
     * 
     * @param $cacheKey
     * @return void
     */
    protected function removeInCacheByKey($cacheKey)
    {
        if ($this->cache !== null) {
            $this->cache->delete($cacheKey);
        }
        $cacheFile = DIR_FS_CATALOG . 'cache/' . static::CACHE_FILE;
        if (file_exists($cacheFile)) {
            unlink($cacheFile);
        }
    }
}
